<?php
require_once 'config/database.php';

// Simple authentication check
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user_id'];

// Get user info
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$user = $stmt->fetch();

if (!$user) {
    session_destroy();
    header("Location: login.php");
    exit;
}

// Get statistics with error handling
try {
    // Total invoices
    $total_invoices_stmt = $pdo->prepare("SELECT COUNT(*) as count FROM invoices WHERE user_id = ?");
    $total_invoices_stmt->execute([$user_id]);
    $total_invoices = $total_invoices_stmt->fetch()['count'];

    // Pending invoices
    $pending_invoices_stmt = $pdo->prepare("SELECT COUNT(*) as count FROM invoices WHERE user_id = ? AND status IN ('sent', 'overdue')");
    $pending_invoices_stmt->execute([$user_id]);
    $pending_invoices = $pending_invoices_stmt->fetch()['count'];

    // Total revenue
    $total_revenue_stmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount), 0) as total FROM invoices WHERE user_id = ? AND status = 'paid'");
    $total_revenue_stmt->execute([$user_id]);
    $total_revenue = $total_revenue_stmt->fetch()['total'];

    // Pending amount
    $pending_amount_stmt = $pdo->prepare("SELECT COALESCE(SUM(total_amount), 0) as total FROM invoices WHERE user_id = ? AND status IN ('sent', 'overdue')");
    $pending_amount_stmt->execute([$user_id]);
    $pending_amount = $pending_amount_stmt->fetch()['total'];

    // Recent invoices
    $recent_invoices_stmt = $pdo->prepare("SELECT * FROM invoices WHERE user_id = ? ORDER BY created_at DESC LIMIT 5");
    $recent_invoices_stmt->execute([$user_id]);
    $recent_invoices = $recent_invoices_stmt->fetchAll();

    // Admin statistics
    if ($user['user_type'] == 'admin') {
        $total_users_stmt = $pdo->query("SELECT COUNT(*) as count FROM users");
        $total_users = $total_users_stmt->fetch()['count'];

        $total_products_stmt = $pdo->query("SELECT COUNT(*) as count FROM products");
        $total_products = $total_products_stmt->fetch()['count'];

        $total_invoices_all_stmt = $pdo->query("SELECT COUNT(*) as count FROM invoices");
        $total_invoices_all = $total_invoices_all_stmt->fetch()['count'];
    }
} catch (PDOException $e) {
    // Handle database errors gracefully
    $error = "Database error: " . $e->getMessage();
    $total_invoices = 0;
    $pending_invoices = 0;
    $total_revenue = 0;
    $pending_amount = 0;
    $recent_invoices = [];
}
?>

<?php include 'includes/header.php'; ?>

<div class="container mt-4">
    <?php if (isset($error)): ?>
        <div class="alert alert-danger"><?php echo $error; ?></div>
    <?php endif; ?>

    <div class="row">
        <div class="col-12">
            <h2>Dashboard</h2>
            <p class="text-muted">Welcome back, <?php echo htmlspecialchars($user['username']); ?>!</p>
        </div>
    </div>

    <div class="row">
        <div class="col-md-3 mb-4">
            <div class="card stat-card">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h3><?php echo $total_invoices; ?></h3>
                            <p class="mb-0">Total Invoices</p>
                        </div>
                        <div class="stat-icon">
                            <i class="fas fa-file-invoice"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-md-3 mb-4">
            <div class="card stat-card">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h3><?php echo $pending_invoices; ?></h3>
                            <p class="mb-0">Pending Invoices</p>
                        </div>
                        <div class="stat-icon">
                            <i class="fas fa-clock"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-md-3 mb-4">
            <div class="card stat-card">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h3>₹<?php echo number_format($total_revenue, 2); ?></h3>
                            <p class="mb-0">Total Revenue</p>
                        </div>
                        <div class="stat-icon">
                            <i class="fas fa-rupee-sign"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <div class="col-md-3 mb-4">
            <div class="card stat-card">
                <div class="card-body">
                    <div class="d-flex justify-content-between">
                        <div>
                            <h3>₹<?php echo number_format($pending_amount, 2); ?></h3>
                            <p class="mb-0">Pending Amount</p>
                        </div>
                        <div class="stat-icon">
                            <i class="fas fa-money-bill-wave"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <?php if ($user['user_type'] == 'admin'): ?>
    <div class="row mt-2">
        <div class="col-md-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5>Admin Statistics</h5>
                    <span class="badge bg-success">Admin User</span>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card bg-admin">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h3><?php echo $total_users; ?></h3>
                                            <p class="mb-0">Total Users</p>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="fas fa-users"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card bg-admin">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h3><?php echo $total_products; ?></h3>
                                            <p class="mb-0">Total Products</p>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="fas fa-boxes"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4 mb-3">
                            <div class="card stat-card bg-admin">
                                <div class="card-body">
                                    <div class="d-flex justify-content-between">
                                        <div>
                                            <h3><?php echo $total_invoices_all; ?></h3>
                                            <p class="mb-0">All Invoices</p>
                                        </div>
                                        <div class="stat-icon">
                                            <i class="fas fa-file-invoice-dollar"></i>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <div class="row mt-4">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h5>Recent Invoices</h5>
                </div>
                <div class="card-body">
                    <?php if (!empty($recent_invoices)): ?>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Invoice #</th>
                                        <th>Client</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_invoices as $invoice): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($invoice['invoice_number']); ?></td>
                                            <td><?php echo htmlspecialchars($invoice['client_name']); ?></td>
                                            <td>₹<?php echo number_format($invoice['total_amount'], 2); ?></td>
                                            <td>
                                                <span class="badge bg-<?php 
                                                    echo $invoice['status'] == 'paid' ? 'success' : 
                                                         ($invoice['status'] == 'sent' ? 'warning' : 
                                                         ($invoice['status'] == 'draft' ? 'secondary' : 'danger')); 
                                                ?>">
                                                    <?php echo ucfirst($invoice['status']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo date('d M Y', strtotime($invoice['invoice_date'])); ?></td>
                                            <td>
                                                <a href="modules/invoices/view.php?id=<?php echo $invoice['id']; ?>" class="btn btn-sm btn-primary">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php else: ?>
                        <p class="text-center text-muted">No invoices found. <a href="modules/invoices/create.php">Create your first invoice</a></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <div class="col-md-4">
            <div class="card">
                <div class="card-header">
                    <h5>Quick Actions</h5>
                </div>
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <a href="modules/invoices/create.php" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Create New Invoice
                        </a>
                        <a href="modules/products/manage.php" class="btn btn-outline-primary">
                            <i class="fas fa-box"></i> Manage Products
                        </a>
                        <a href="modules/invoices/" class="btn btn-outline-primary">
                            <i class="fas fa-list"></i> View All Invoices
                        </a>
                        <?php if ($user['user_type'] == 'admin'): ?>
                            <a href="#" class="btn btn-outline-success">
                                <i class="fas fa-cog"></i> Admin Settings
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="card mt-4">
                <div class="card-header">
                    <h5>Account Info</h5>
                </div>
                <div class="card-body">
                    <p><strong>Username:</strong> <?php echo htmlspecialchars($user['username']); ?></p>
                    <p><strong>Email:</strong> <?php echo htmlspecialchars($user['email']); ?></p>
                    <p><strong>Company:</strong> <?php echo htmlspecialchars($user['company_name'] ?: 'Not set'); ?></p>
                    <p><strong>User Type:</strong> 
                        <span class="badge bg-<?php echo $user['user_type'] == 'admin' ? 'success' : 'primary'; ?>">
                            <?php echo ucfirst($user['user_type']); ?>
                        </span>
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>